definition module StdControl


//	********************************************************************************
//	Clean Standard Object I/O library, version 1.0.1
//	
//	StdControl specifies all control operations.
//	Changing controls in a window/dialogue requires a *WState.
//	Reading the status of controls requires a WState.
//	********************************************************************************


import	StdControlDef, StdMaybe
from	iostate		import IOSt


::	WState


getWindow				:: !Id !(IOSt .l .p) -> (!Maybe WState, !IOSt .l .p)
/*	getWindow returns a read-only WState for the indicated window.
	In case the indicated window does not exist Nothing is returned.
*/

setWindow				:: !Id ![IdFun *WState] !(IOSt .l .p) -> IOSt .l .p
/*	Apply the control changing functions to the current state of the indicated 
	window. 
	In case the indicated window does not exist nothing happens.
*/


/*	Functions that change the state of controls.
	When applied to unknown Ids these functions have no effect.
*/
showControls			:: ![Id]							!*WState -> *WState
hideControls			:: ![Id]							!*WState -> *WState
/*	(show/hide)Controls makes the indicated controls visible/invisible. 
	Hiding a control overrides the visibility of its elements, which become 
		invisible. 
	Showing a hidden control re-establishes the visibility state of its elements.
*/

enableControls			:: ![Id]							!*WState -> *WState
disableControls			:: ![Id]							!*WState -> *WState
/*	(en/dis)ableControls (en/dis)ables the indicated controls.
	Disabling a control overrides the SelectStates of its elements, which become 
		unselectable.
	Enabling a disabled control re-establishes the SelectStates of its elements.
*/

markCheckControlItems	:: !Id ![Index]						!*WState -> *WState
unmarkCheckControlItems	:: !Id ![Index]						!*WState -> *WState
/*	(unm/m)arkCheckControlItems unmarks/marks the indicated check items of the given
	CheckControl. Indices range from 1 to the number of check items. Illegal indices
	are ignored.
*/

selectRadioControlItem	:: !Id  !Index						!*WState -> *WState
/*	selectRadioControlItem marks the indicated radio item of a RadioControl, causing
	the mark of the previously marked radio item to disappear. The item is given by 
	the Id of the RadioControl and its index position (counted from 1). 
*/

selectPopUpControlItem	:: !Id  !Index						!*WState -> *WState
/*	selectPopUpControlItem marks the indicated popup item of a PopUpControl, causing
	the mark of the previously marked popup item to disappear. The item is given by 
	the Id of the PopUpControl and its index position (counted from 1).
*/

moveControlViewFrame	:: !Id Vector						!*WState -> *WState
/*	moveControlViewFrame moves the orientation of the CompoundControl over the given
	vector, and updates the control if necessary. The control frame is not moved 
	outside the ViewDomain of the control. MoveControlViewFrame has no effect if the
	indicated control has no ControlDomain attribute.
*/

setControlTexts			:: ![(Id,String)]					!*WState -> *WState
/*	setControlTexts sets the text of the indicated (Text/Edit/Button)Controls. 
	If the indicated control is a (Text/Button)Control, then AltKey are interpreted 
	by the system.
	If the indicated control is an EditControl, then the text is taken as it is.
*/

setEditControlCursor	:: !Id !Int							!*WState -> *WState
/*	setEditControlCursor sets the cursor at position @2 of the current content of 
	the EditControl.
	In case @2<0, then the cursor is set at the start of the current content.
	In case @2>size content, then the cursor is set at the end of the current 
	content.
*/

setControlLooks			:: ![(Id,Bool,Look)]				!*WState -> *WState
/*	setControlLooks applied to a CompoundControl turns it into a non-transparant 
	CompoundControl.
	Setting the Look only redraws the indicated controls if the corresponding 
	Boolean is True. 
*/

setSliderStates			:: ![(Id,SliderState->SliderState)]	!*WState -> *WState
setSliderThumbs			:: ![(Id,Int)]						!*WState -> *WState
/*	setSliderStates
		applies the function to the current SliderState of the indicated 
		SliderControl and redraws the settings if necessary.
	setSliderThumbs
		sets the new thumb value of the indicated SliderControl and redraws the 
		settings if necessary.
*/


drawInControl			:: !Id ![DrawFunction]				!*WState -> *WState
/*	Draw in a (Custom(Button)/Compound)Control. If the CompoundControl is 
	transparant then this operation has no effect.
*/


getControlTypes			::			!WState -> [(ControlType,Maybe Id)]
getCompoundTypes		:: !Id		!WState -> [(ControlType,Maybe Id)]
/*	getControlTypes
		yields the list of ControlTypes of the component controls of this window. 
	getCompoundTypes
		yields the list of ControlTypes of the component controls of this 
		CompoundControl. 
	For both functions (Just id) is yielded if the component control has a 
	(ControlId id) attribute, and Nothing otherwise. Component controls are not 
	collected recursively through CompoundControls.
	If the indicated CompoundControl is not a CompoundControl, then [] is yielded.
*/

getControlLayouts		:: ![Id] !WState -> [(Bool,(Maybe ItemPos,ItemOffset))]
																// (Nothing,zero)
getControlViewSizes		:: ![Id] !WState -> [(Bool,Size)]				// zero
getControlSelectStates	:: ![Id] !WState -> [(Bool,SelectState)]		// Able
getControlShowStates	:: ![Id] !WState -> [(Bool,Bool)]				// False
getControlTexts			:: ![Id] !WState -> [(Bool,Maybe String)]		// Nothing
getControlNrLines		:: ![Id] !WState -> [(Bool,Maybe NrLines)]		// Nothing
getControlLooks			:: ![Id] !WState -> [(Bool,Maybe Look)]			// Nothing
getControlMinimumSizes	:: ![Id] !WState -> [(Bool,Maybe Size)]			// Nothing
getControlResizes		:: ![Id] !WState -> [(Bool,Maybe ControlResizeFunction)]
																		// Nothing
getRadioControlItems	:: ![Id] !WState -> [(Bool,Maybe [TextLine])]	// Nothing
getRadioControlSelection:: ![Id] !WState -> [(Bool,Maybe Index)]		// Nothing
getCheckControlItems	:: ![Id] !WState -> [(Bool,Maybe [TextLine])]	// Nothing
getCheckControlSelection:: ![Id] !WState -> [(Bool,Maybe [Index])]		// Nothing
getPopUpControlItems	:: ![Id] !WState -> [(Bool,Maybe [TextLine])]	// Nothing
getPopUpControlSelection:: ![Id] !WState -> [(Bool,Maybe Index)]		// Nothing
getSliderDirections		:: ![Id] !WState -> [(Bool,Maybe Direction)]	// Nothing
getSliderStates			:: ![Id] !WState -> [(Bool,Maybe SliderState)]	// Nothing
getControlViewFrames	:: ![Id] !WState -> [(Bool,Maybe ViewFrame)]	// Nothing
getControlViewDomains	:: ![Id] !WState -> [(Bool,Maybe ViewDomain)]	// Nothing
getControlItemSpaces	:: ![Id] !WState -> [(Bool,Maybe (Int,Int))]	// Nothing
getControlMargins		:: ![Id] !WState -> [(Bool,Maybe ((Int,Int),(Int,Int)))]
																		// Nothing

/*	Functions that return the current state of controls.
	The result list is of equal length as the argument Id list. Each result list 
	element corresponds in order with the argument Id list. Of each element the 
	first Boolean result is False in case of invalid Ids (if so dummy values are 
	returned - see comment).
	Important: controls with no ControlId attribute, or illegal ids, can not be 
	found in the WState!
	
	getControlLayouts
		Yields (Just ControlPos) if the indicated control had a ControlPos attribute 
		and Nothing otherwise. The ItemOffset offset is the exact current location 
		of the indicated control (LeftTop,offset). 
	getControlViewSizes
		Yields the current view frame size of the indicated control. Note that for 
		any control other than the CompoundControl this is the exact size of the 
		control.
	getControlSelectStates
		Yields the current SelectState of the indicated control.
	getControlShowStates
		Yields True if the indicated control is visible, and False otherwise. 
	getControlTexts
		Yields (Just text) of the indicated (Text/Edit/Button)Control. 
		If the control is not such a control, then Nothing is yielded. 
	getControlNrLines
		Yields (Just nrlines) of the indicated EditControl. 
		If the control is not such a control, then Nothing is yielded.
	getControlLooks
		Yields the Look of the indicated (Custom/CustomButton/Compound)Control. 
		If the control is not such a control, or is a transparant CompoundControl, 
		then Nothing is yielded.
	getControlMinimumSizes
		Yields (Just minimumsize) if the indicated control had a ControlMinimumSize 
		attribute and Nothing otherwise. 
	getControlResizes
		Yields (Just resizefunction) if the indicated control had a ControlResize 
		attribute and Nothing otherwise.
	getRadioControlItems
		Yields the TextLines of the items of the indicated RadioControl.
		If the control is not such a control, then Nothing is yielded.
	getRadioControlSelection
		Yields the index of the selected radio item of the indicated RadioControl. 
		If the control is not such a control, then Nothing is yielded.
	getCheckControlItems
		Yields the TextLines of the items of the indicated CheckControl. 
		If the control is not such a control, then Nothing is yielded.
	getCheckControlSelection
		Yields the indices of the selected checkitems of the indicated CheckControl. 
		If the control is not such a control, then Nothing is yielded.
	getPopUpControlItems
		Yields the TextLines of the items of the indicated PopUpControl. 
		If the control is not such a control, then Nothing is yielded.
	getPopUpControlSelection
		Yields the Index of the indicated PopUpControl.
		If the control is not such a control, then Nothing is yielded.
	getSliderDirections
		Yields (Just Direction) of the indicated SliderControl. 
		If the control is not such a control, then Nothing is yielded.
	getSliderStates
		Yields (Just SliderState) of the indicated SliderControl. 
		If the control is not such a control, then Nothing is yielded. 
	getControlViewFrames
		Yields (Just ViewFrame) of the indicated CompoundControl. 
		If the control is not such a control, then Nothing is yielded.
	getControlViewDomains
		Yields (Just ViewDomain) of the indicated CompoundControl.
		If the control is not such a control, then Nothing is yielded.
	getControlItemSpaces
		Yields (Just (horizontal space,vertical space)) of the indicated 
		CompoundControl.
		If the control is not such a control, then Nothing is yielded. 
	getControlMargins
		Yields (Just (ControlHMargin,ControlVMargin)) of the indicated
		CompoundControl. 
		If the control is not such a control, then Nothing is yielded.
*/
